<?php
// get_schools.php - COMPLETE SOLUTION

// Ensure no previous output
if (ob_get_level()) ob_end_clean();
ob_start();

// Set headers first
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate');
header('Pragma: no-cache');

// Disable error display on production
ini_set('display_errors', '0');
error_reporting(0);

// Define clean response function
function json_response($data, $status = 200) {
    http_response_code($status);
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_NUMERIC_CHECK);
    exit;
}

try {
    // Validate input
    if (!isset($_GET['directorate_id']) || !is_numeric($_GET['directorate_id'])) {
        json_response(['error' => 'Invalid directorate ID'], 400);
    }

    $directorate_id = (int)$_GET['directorate_id'];
    if ($directorate_id <= 0) {
        json_response(['error' => 'Directorate ID must be positive'], 400);
    }

    // Include config with absolute path
    require __DIR__ . '/config.php';

    // Verify database connection
    if (!($conn instanceof mysqli) || $conn->connect_error) {
        throw new Exception('Database connection failed');
    }

    // Prepare statement with explicit error handling
    $sql = "SELECT SchoolID, SchoolName FROM schools WHERE DirectorateID = ?";
    if (!($stmt = $conn->prepare($sql))) {
        throw new Exception("Prepare failed: " . $conn->error);
    }

    // Bind and execute
    if (!$stmt->bind_param("i", $directorate_id) || !$stmt->execute()) {
        throw new Exception("Query failed: " . $stmt->error);
    }

    // Get results
    $result = $stmt->get_result();
    if (!$result) {
        throw new Exception("Result set failed: " . $stmt->error);
    }

    // Fetch data
    $schools = [];
    while ($row = $result->fetch_assoc()) {
        $schools[] = [
            'SchoolID' => (int)$row['SchoolID'],
            'SchoolName' => $row['SchoolName']
        ];
    }

    // Clean output and return
    ob_end_clean();
    json_response([
        'success' => true,
        'data' => $schools,
        'count' => count($schools)
    ]);

} catch (Exception $e) {
    // Log error securely
    error_log(date('[Y-m-d H:i:s]') . " get_schools.php Error: " . $e->getMessage());
    
    // Return clean error
    ob_end_clean();
    json_response([
        'success' => false,
        'error' => 'Service unavailable',
        'code' => 500
    ], 500);
}